<?php

namespace App\Http\Controllers;

use App\Models\CollegeConfig;
use App\Models\Mark;
use App\Models\MarkConfig;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use Barryvdh\Snappy\Facades\SnappyPdf as PDF;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\MarksImport;
use Carbon\Carbon;

class MarkController extends Controller
{
    public $user;
    function __construct()
    {
        $this->user = Auth::user();
    }

    public function index()
    {
        $exams = Mark::where('college_id', $this->user->college_id)
        ->where('status', 1)
        ->whereDate('start_date', '<=', Carbon::today())  
        ->whereDate('end_date', '>=', Carbon::today()) 
        ->get();
        $college_config = CollegeConfig::where('college_id', $this->user->college_id)->get();
        return view('mark-upload.index', compact('exams', 'college_config'));
    }

    public function studentIndex()
    {
        $exams = Mark::where('college_id', $this->user->college_id)
        ->where('status', 1)
        ->whereDate('start_date', '<=', Carbon::today())  
        ->whereDate('end_date', '>=', Carbon::today()) 
        ->get();
        $college_config = CollegeConfig::where('college_id', $this->user->college_id)->get();
        return view('mark-upload.index-std', compact('exams', 'college_config'));
    }

    public function resultIndex()
    {
        $exams = Mark::where('status', 1)
        ->whereDate('start_date', '<=', Carbon::today())  
        ->whereDate('end_date', '>=', Carbon::today()) 
        ->get();
        return view('mark-upload.std-mark-report', compact('exams'));
    }

    public function import(Request $request)
    {
        $request->validate([
            'exam_name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date',
            'mark_file' => 'required|file|mimes:xlsx,xls,csv',
        ]);

        $file = $request->file('mark_file');
        $college_name = CollegeConfig::where('college_id', $this->user->college_id)->first()->college_name;
        $mark = Mark::create([
            'college_id' => $this->user->college_id,
            'college_name' => $college_name,
            'exam_name' => $request->exam_name,
            'status' => $request->status,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'exam_type' => $request->result_type,
        ]);
        $college_config = CollegeConfig::select('config_key', 'config_value')
            ->where('college_id', $this->user->college_id)->get();

        Excel::import(new MarksImport($request->all(), $mark, $college_config), $file);

        return redirect()->route('mark-upload')->with('success', 'Marks uploaded successfully!');
    }

    public function getMarksData(Request $request)
    {

        $config_data = DB::table('college_config')
            ->select('config_key', 'config_value')
            ->where('college_id', $this->user->college_id)
            ->get();

        $course = '';
        $semester = '';
        $subject = '';

        $config_data->each(function ($config) use (&$course, &$semester, &$subject) {
            if ($config->config_value == 'Course Name') {
                $course = $config->config_key;
            } elseif ($config->config_value == 'Semester') {
                $semester = $config->config_key;
            } elseif ($config->config_value == 'Subject') {
                $subject = $config->config_key;
            }
        });

        $marks = MarkConfig::
            when($request->has('exam') && $request->exam != '', function ($query) use ($request) {
                return $query->where('mark_id', $request->exam);
            })
            ->when($this->user->college_id, function ($query) use ($request) {
                return $query->whereHas('mark', function ($query) use ($request) {
                    $query->where('college_id', $this->user->college_id);
                });
            })
            ->when($request->has('semester') && $request->semester != '', function ($query) use ($request, $semester) {
                return $query->where($semester, $request->semester);
            })
            ->when($request->has('result') && $request->result != '', function ($query) use ($request) {
                return $query->whereHas('mark', function ($query) use ($request) {
                    $query->where('exam_type', $request->result);
                });
            });

        return DataTables::of($marks)
            ->make(true);
    }


    public function getStdMark(Request $request)
    {

        if (empty($request->exam) || $request->exam == '') {
            return response()->json(['error' => 'Please enter the Register No.'], 500);
        }
        $config_data = DB::table('college_config')
            ->select('config_key', 'config_value')
            ->where('college_id', $this->user->college_id)
            ->get();

        $reg_no = '';
        $semester = '';


        $config_data->each(function ($config) use (&$reg_no, &$semester) {
            if ($config->config_value == 'Register Number') {
                $reg_no = $config->config_key;
            } elseif ($config->config_value == 'Semester') {
                $semester = $config->config_key;
            }
        });

        $marks = MarkConfig::
            when($request->has('exam') && $request->exam != '', function ($query) use ($request, $reg_no) {
                return $query->where($reg_no, $request->exam);
            })
            ->when($request->has('semester') && $request->semester != '', function ($query) use ($request, $semester) {
                return $query->where($semester, $request->semester);
            })
            ->when($this->user->college_id, function ($query) use ($request) {
                return $query->whereHas('mark', function ($query) use ($request) {
                    $query->where('college_id', $this->user->college_id);
                });
            });

        return DataTables::of($marks)
            ->make(true);
    }

    public function generatePDF(Request $request)
    {
        $rules = [
            'exam' => 'required',
            'register_number' => 'required',
            'semester' => 'required',
        ];

        $request->validate($rules);
        $config_data = DB::table('college_config')
            ->select('config_key', 'config_value')
            ->get();
        $college = DB::table('college_info')->first();
        if (empty($config_data)) {
            return redirect()->back()->withInput()->with('error', 'No results found');
        }

        $reg_no = '';
        $semester = '';


        $config_data->each(function ($config) use (&$reg_no, &$semester) {
            if ($config->config_value == 'Register Number') {
                $reg_no = $config->config_key;
            } elseif ($config->config_value == 'Semester') {
                $semester = $config->config_key;
            }
        });

        $marks = MarkConfig::
            when($request->has('register_number') && $request->register_number != '', function ($query) use ($request, $reg_no) {
                return $query->where($reg_no, $request->register_number);
            })
            ->when($request->has('semester') && $request->semester != '', function ($query) use ($request, $semester) {
                return $query->where($semester, $request->semester);
            })
            ->when($request->has('exam') && $request->exam != '', function ($query) use ($request) {
                return $query->where('mark_id', $request->exam);
            })
            ->get();
        $mark_first = MarkConfig::
            when($request->has('register_number') && $request->register_number != '', function ($query) use ($request, $reg_no) {
                return $query->where($reg_no, $request->register_number);
            })
            ->when($request->has('semester') && $request->semester != '', function ($query) use ($request, $semester) {
                return $query->where($semester, $request->semester);
            })
            ->when($request->has('exam') && $request->exam != '', function ($query) use ($request) {
                return $query->where('mark_id', $request->exam);
            })
            ->first();
        $config_hide = [];
        $config_view = [];
        $mark_key = '';
        $std_name_key = '';
        $reg_no_key = '';
        $course_key = '';
        $result_key = '';
        foreach ($config_data as $data) {
            if ($data->config_value == "Student Name" || $data->config_value == "Register Number" || $data->config_value == "Course Name" || $data->config_value == null) {
                $config_hide[] = $data;
            } else {
                $config_view[] = $data;
            }

            if ($data->config_value == 'Actual Marks') {
                $mark_key = $data->config_key;
            } else if ($data->config_value == "Student Name") {
                $std_name_key = $data->config_key;
            } else if ($data->config_value == "Register Number") {
                $reg_no_key = $data->config_key;
            } else if ($data->config_value == "Course Name") {
                $course_key = $data->config_key;
            }else if ($data->config_value == "Result") {
                $result_key = $data->config_key;
            }

        }

        $hidden_keys = collect($config_hide)->pluck('config_key')->toArray();

        foreach ($marks as $mark) {
            foreach ($config_hide as $data) {
                if (isset($mark->{$data->config_key})) {
                    unset($mark->{$data->config_key});
                }
            }
        }
        $total_marks = $marks->sum($mark_key);

        foreach ($marks as $mark) {
            foreach ($mark->getAttributes() as $key => $value) {
                if ($key == 'mark_id' || $key == 'id' || $key == 'created_at' || $key == 'updated_at') {
                    unset($mark->{$key});
                }
            }
        }

        if (empty($marks) || empty($mark_first) || empty($config_view)) {
            return redirect()->back()->withInput()->with('error', 'No results found');
        }

        $logo_path = public_path('logos/' . $college->logo);
        if (file_exists($logo_path)) {
            $logo_data = base64_encode(file_get_contents($logo_path));
            $mime_type = mime_content_type($logo_path);
            $logo_base64 = 'data:' . $mime_type . ';base64,' . $logo_data;
        } else {
            $logo_base64 = '';
        }
        $college_logo = $logo_base64;

        $pdf = PDF::loadView('mark-upload.result-pdf', compact(
            'total_marks',
            'college',
            'marks',
            'mark_first',
            'config_view',
            'college_logo',
            'std_name_key',
            'reg_no_key',
            'course_key',
            'result_key'
        ));
        $pdf->setOption('orientation', 'landscape');
        return $pdf->stream('semester_results_report.pdf');
    }


}
