<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\BranchModel;
use App\Models\ClientModel;
use App\Models\InvoiceItemModel;
use App\Models\InvoiceItemTaxModel;
use App\Models\InvoiceModel;
use App\Models\LoyaltyTransactionModel;
use App\Models\OrdersModel;
use App\Models\PaymentsModel;
use App\Models\ProductModel;
use App\Models\QuotesModel;
use App\Models\TaxesModel;
use App\Models\User;
use Auth;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mpdf\Mpdf;
use Storage;

class MobInvoiceController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        if ($user->user_flg == "C") {
            $query = InvoiceModel::with([
                'client.user:id,name,phone',
                'createdBy:id,name',
                'invoiceItems:id,invoice_id,product_id,product_name,quantity,price,margin_price,total',
                'invoiceItems.invoiceItemTaxes:id,invoice_item_id,tax_id,tax',
                'payments:id,invoice_id,amount,total_amount,payment_mode,created_at',
            ])
                ->select([
                    'id',
                    'invoice_id',
                    'client_id',
                    'branch_id',
                    'invoice_date',
                    'due_date',
                    'gst',
                    'amount',
                    'final_amount',
                    'discount_type',
                    'discount',
                    'note',
                    'term',
                    'status',
                    'created_by',
                    'created_at',
                    'updated_at',
                ])
                ->where('branch_id', $request->branch_id)
                ->where('oprntl_flag', 'A')
                ->where('client_id', $user->client->id);
        } else {
            $query = InvoiceModel::with([
                'client.user:id,name,phone',
                'createdBy:id,name',
                'invoiceItems:id,invoice_id,product_id,product_name,quantity,price,margin_price,total',
                'invoiceItems.invoiceItemTaxes:id,invoice_item_id,tax_id,tax',
                'payments:id,invoice_id,amount,total_amount,payment_mode,created_at',
            ])
                ->select([
                    'id',
                    'invoice_id',
                    'client_id',
                    'branch_id',
                    'invoice_date',
                    'due_date',
                    'gst',
                    'amount',
                    'final_amount',
                    'discount_type',
                    'discount',
                    'note',
                    'term',
                    'status',
                    'created_by',
                    'created_at',
                    'updated_at',
                ])
                ->where('branch_id', $request->branch_id)
                ->where('oprntl_flag', 'A');
        }


        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('invoice_id', 'like', "%$searchTerm%")
                    ->orWhereHas('client.user', function ($q2) use ($searchTerm) {
                        $q2->where('name', 'like', "%$searchTerm%")
                            ->orWhere('phone', 'like', "%$searchTerm%");
                    });
            });
        }

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $query->whereBetween('invoice_date', [
                $request->from_date,
                $request->to_date
            ]);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $perPage = $request->get('per_page', 10); // default 10 per page
        $invoices = $query->orderBy('id', 'desc')->paginate($perPage);

        return response()->json([
            'invoices' => $invoices
        ]);
    }


    public function show(Request $request, $id)
    {
        $invoices = InvoiceModel::with([
            'client.user:id,name,phone',
            'createdBy:id,name',
            'invoiceItems:id,invoice_id,product_id,product_name,quantity,price,margin_price,total',
            'invoiceItems.invoiceItemTaxes:id,invoice_item_id,tax_id,tax',
            'payments:id,invoice_id,amount,total_amount,payment_mode,created_at',
        ])
            ->select([
                'id',
                'invoice_id',
                'client_id',
                'branch_id',
                'invoice_date',
                'due_date',
                'gst',
                'amount',
                'final_amount',
                'discount_type',
                'discount',
                'note',
                'term',
                'status',
                'created_by',
                'created_at',
                'updated_at',
            ])
            ->where('branch_id', $request->branch_id)
            ->where('oprntl_flag', 'A')
            ->where('id', $id)
            ->orderBy('id', 'desc')
            ->first();
        return response()->json([
            'invoices' => $invoices
        ]);
    }
    public function getTaxes(Request $request)
    {
        $tax = TaxesModel::where('show_flg', 'Y')->where('branch_id', $request->branch_id)->get();
        return response()->json($tax);
    }

    public function store(Request $request)
    {
        $rules = [
            'invoice_date' => 'required|date',
            'due_date' => 'required|date',
            'sub_total' => 'required|numeric',
            'total_amt' => 'nullable|numeric',
            'discount_type' => 'nullable|numeric',
            'note' => 'nullable|string',
            'term' => 'nullable|string',
            'gst' => 'required|in:Y,N',
            'status' => 'required|in:1,2,3',
            'dynamicFields.*.product' => 'required|exists:products,id',
            'dynamicFields.*.qty' => 'required|numeric',
            'dynamicFields.*.unit_price' => 'required|numeric',
            'dynamicFields.*.amount' => 'required|numeric',
        ];

        if (!empty($request->discount_type)) {
            $rules['discount'] = 'required|numeric';
        }

        if (empty($request->client_num)) {
            $rules['client_num'] = 'required';
        }

        $user = User::where('phone', $request->client_num)->first();
        if (!$user) {
            $rules['client_name'] = 'required|string';
        }

        if ($request->gst == "Y") {
            $rules['dynamicFields.*.tax'] = 'required|array';
            $rules['dynamicFields.*.tax.*'] = 'required|numeric|gt:0';
        }

        $request->validate($rules);

        if (($request->cash + $request->upi + $request->online) > $request->total_amt) {
            return response()->json(['error' => 'Total of cash, UPI, and online payments cannot exceed total amount.'], 400);
        }

        DB::beginTransaction();

        try {
            $client_id = $this->saveClient($request);

            $lastInvoice = InvoiceModel::latest('id')->first();
            $invoice_id = $lastInvoice
                ? explode('-', $lastInvoice->invoice_id)[0] . '-' . str_pad(((int) explode('-', $lastInvoice->invoice_id)[1]) + 1, 3, '0', STR_PAD_LEFT)
                : 'INV-001';

            $branch_id = auth()->user()->branch_id ?? $request->branch_id;

            $clientPoints = ClientModel::findOrFail($client_id);
            $loyalty = BranchModel::find($branch_id);

            if ($request->redeem_points) {
                $current_point = $request->available_point - $request->redeem_points;

                LoyaltyTransactionModel::create([
                    'client_id' => $client_id,
                    'points' => $request->redeem_points,
                    'type' => 'redeem',
                    'reference_invoice' => $invoice_id,
                    'created_by' => auth()->id(),
                ]);
                $clientPoints->update(['loyalty_points' => $current_point]);
            }

            if ($loyalty->loyalty_config == 'A') {
                $earned = floor($request->total_amt / $loyalty->loyalty_amount);
                $clientPoints->increment('loyalty_points', $earned);
                LoyaltyTransactionModel::create([
                    'client_id' => $client_id,
                    'points' => $earned,
                    'type' => 'earn',
                    'reference_invoice' => $invoice_id,
                    'created_by' => auth()->id(),
                ]);
            }
            $paidAmount = round($request->cash + $request->upi + $request->online);
            $totalAmount = round($request->total_amt);

            if ($paidAmount === 0) {
                $status = 1; // unpaid
            } elseif ($paidAmount >= $totalAmount) {
                $status = 2; // fully paid
            } else {
                $status = 3; // partially paid
            }
            $invoice = InvoiceModel::create([
                'invoice_id' => $invoice_id,
                'client_id' => $client_id,
                'invoice_date' => $request->invoice_date,
                'due_date' => $request->due_date,
                'status' => $status,
                'currency_id' => 3,
                'recurring' => $request->recurring ?? false,
                'recurring_cycle' => $request->recurring_cycle,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amt,
                'discount_type' => $request->discount_type,
                'discount' => $request->discount,
                'note' => $request->note,
                'term' => $request->term,
                'gst' => $request->gst,
                'branch_id' => $branch_id,
                'created_by' => auth()->id(),
            ]);

            foreach ($request->dynamicFields as $field) {
                $item = InvoiceItemModel::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $field['product'],
                    'product_name' => ProductModel::find($field['product'])->name,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'margin_price' => ProductModel::find($field['product'])->margin_price,
                    'total' => $field['amount'],
                ]);

                $product = ProductModel::findOrFail($field['product']);
                $product->decrement('qty', (int) $field['qty']);

                if ($request->gst == "Y") {
                    foreach ($field['tax'] as $taxId) {
                        InvoiceItemTaxModel::create([
                            'invoice_item_id' => $item->id,
                            'tax_id' => $taxId,
                            'tax' => TaxesModel::find($taxId)->value,
                        ]);
                    }
                }
            }

            // Payment logic
            foreach (['upi' => 1, 'online' => 2, 'cash' => 3] as $method => $mode) {
                if (!empty($request->$method)) {
                    PaymentsModel::create([
                        'invoice_id' => $invoice->id,
                        'amount' => $request->$method,
                        'total_amount' => $request->total_amt,
                        'payment_mode' => $mode
                    ]);
                }
            }

            if (!empty($request->order_id)) {
                $order = OrdersModel::findOrFail($request->order_id);
                $order->oprntl_flag = 'C';
                $order->status = 3;
                $order->save();
            }

            if (!empty($request->quote_id)) {
                $quote = QuotesModel::findOrFail($request->quote_id);
                $quote->oprntl_flag = 'C';
                $quote->status = 3;
                $quote->save();
            }

            DB::commit();

            return response()->json(['message' => 'Invoice created successfully', 'data' => $invoice->id], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error("Invoice Store API Error: " . $e->getMessage());
            return response()->json(['error' => 'Server error while creating invoice'], 500);
        }
    }
    public function update(Request $request, $id)
    {
        if ($id == 0) {
            return $this->store($request);
        }
        $rules = [
            'invoice_date' => 'required|date',
            'due_date' => 'required|date',
            'sub_total' => 'required|numeric',
            'total_amt' => 'nullable|numeric',
            'discount_type' => 'nullable|numeric',
            'note' => 'nullable|string',
            'term' => 'nullable|string',
            'gst' => 'required|in:Y,N',
            'status' => 'required|in:1,2,3',
            'dynamicFields.*.product' => 'required|exists:products,id',
            'dynamicFields.*.qty' => 'required|numeric',
            'dynamicFields.*.unit_price' => 'required|numeric',
            'dynamicFields.*.amount' => 'required|numeric',
        ];

        if (!empty($request->discount_type)) {
            $rules['discount'] = 'required|numeric';
        }

        if (empty($request->client_num)) {
            $rules['client_num'] = 'required';
        }

        $user = User::where('phone', $request->client_num)->first();
        if (!$user) {
            $rules['client_name'] = 'required|string';
        }

        if ($request->gst == "Y") {
            $rules['dynamicFields.*.tax'] = 'required|array';
            $rules['dynamicFields.*.tax.*'] = 'required|numeric|gt:0';
        }

        $request->validate($rules);

        if (($request->cash + $request->upi + $request->online) > $request->total_amt) {
            return response()->json(['error' => 'Total of cash, UPI, and online payments cannot exceed total amount.'], 400);
        }

        DB::beginTransaction();

        try {
            $client_id = $this->saveClient($request);

            $invoice = InvoiceModel::findOrFail($id);
            $paidAmount = round($request->cash + $request->upi + $request->online);
            $totalAmount = round($request->total_amt);

            if ($paidAmount === 0) {
                $status = 1; // unpaid
            } elseif ($paidAmount >= $totalAmount) {
                $status = 2; // fully paid
            } else {
                $status = 3; // partially paid
            }
            $invoice->update([
                'client_id' => $client_id,
                'invoice_date' => $request->invoice_date,
                'due_date' => $request->due_date,
                'status' => $status,
                'currency_id' => 3,
                'recurring' => $request->recurring ?? false,
                'recurring_cycle' => $request->recurring_cycle,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amt,
                'discount_type' => $request->discount_type,
                'discount' => $request->discount,
                'note' => $request->note,
                'term' => $request->term,
                'gst' => $request->gst,
                'branch_id' => auth()->user()->branch_id ?? $request->branch_id,
                'updated_by' => auth()->id(),
            ]);

            // Remove existing items & taxes
            InvoiceItemModel::where('invoice_id', $invoice->id)->delete();

            foreach ($request->dynamicFields as $field) {
                $item = InvoiceItemModel::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $field['product'],
                    'product_name' => ProductModel::find($field['product'])->name,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'margin_price' => ProductModel::find($field['product'])->margin_price,
                    'total' => $field['amount'],
                ]);

                $product = ProductModel::findOrFail($field['product']);
                $product->decrement('qty', (int) $field['qty']);

                if ($request->gst == "Y" && isset($field['tax'])) {
                    foreach ($field['tax'] as $taxId) {
                        InvoiceItemTaxModel::create([
                            'invoice_item_id' => $item->id,
                            'tax_id' => $taxId,
                            'tax' => TaxesModel::find($taxId)->value,
                        ]);
                    }
                }
            }

            // Remove existing payments and recreate
            PaymentsModel::where('invoice_id', $invoice->id)->delete();

            foreach (['upi' => 1, 'online' => 2, 'cash' => 3] as $method => $mode) {
                if (!empty($request->$method)) {
                    PaymentsModel::create([
                        'invoice_id' => $invoice->id,
                        'amount' => $request->$method,
                        'total_amount' => $request->total_amt,
                        'payment_mode' => $mode
                    ]);
                }
            }

            DB::commit();
            return response()->json(['message' => 'Invoice updated successfully', 'data' => $invoice->id], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error("Invoice Update Error: " . $e->getMessage());
            return response()->json(['error' => 'Failed to update invoice'], 500);
        }
    }

    protected function saveClient($request)
    {
        $user = User::where('phone', $request->client_num)->first();

        if (!empty($user)) {
            $client = ClientModel::where('user_id', $user->id)->first();
            return $client->id;
        } else {
            $randomEmail = Str::random(10) . '@billing.com';
            $user = User::create([
                'name' => $request->client_name,
                'email' => $randomEmail,
                'phone' => $request->client_num,
                'password' => Hash::make('client@123'),
                'user_flg' => 'C',
                'active_status' => 'A',
                'branch_id' => $request->branch_id
            ]);

            $client = ClientModel::create([
                'user_id' => $user->id,
                'branch_id' => $request->branch_id,
                'active_status' => 'A',
            ]);

            return $client->id;
        }
    }

    public function generateInvoice($invoiceId, Request $request)
    {
        try {
            $addedInvoice = InvoiceModel::with(
                'client.user',
                'branch.company',
                'invoiceItems.invoiceItemTaxes',
                'payments.mode',
                'invoiceItems.products.category.unitOfMass'
            )
                ->where('id', $invoiceId)
                ->where('branch_id', $request->branch_id)
                ->first();

            if (!$addedInvoice) {
                return response()->json(['error' => 'Invoice not found'], 404);
            }

            $invoiceItems = $addedInvoice->invoiceItems;
            $products = [];
            $discount = 0;
            $total = 0;

            foreach ($invoiceItems as $item) {
                $gst = $item->invoiceItemTaxes->sum('tax');
                $subtotal = ($item->price * $gst) / 100;
                $totalWithoutGst = $item->quantity * $item->price;
                $totalWithGst = ($item->quantity * $subtotal) + $totalWithoutGst;
                $total += $totalWithGst;

                $products[] = [
                    'name' => $item->products->name ?? '',
                    'code' => $item->products->code ?? '',
                    'quantity' => $item->quantity ?? '',
                    'uom' => $item->products->category->unitOfMass->name ?? '',
                    'unit_price' => $item->price ?? '',
                    'gst' => $gst,
                    'total_price' => $item->quantity * $subtotal,
                ];
            }

            if ($addedInvoice->discount_type == 1) {
                $discount = $addedInvoice->discount;
            } elseif ($addedInvoice->discount_type == 2) {
                $discount = ($total * $addedInvoice->discount) / 100;
            }

            $sumQty = collect($invoiceItems)->sum('quantity');

            $redeem = LoyaltyTransactionModel::where('client_id', $addedInvoice->client_id)
                ->where('reference_invoice', $addedInvoice->invoice_id)
                ->get();

            $redeemed_points = $redeem->where('type', 'redeem')->sum('points');
            $redeemed_value = $redeemed_points * $addedInvoice->branch->point_redemption;

            $data = [
                'companyName' => $addedInvoice->branch->company->name,
                'companyAddress' => $addedInvoice->branch->company->address,
                'contact_no' => $addedInvoice->branch->company->phone,
                'email' => $addedInvoice->branch->company->email,
                'clientName' => $addedInvoice->client->user->name,
                'clientAddress' => $addedInvoice->client->user->location,
                'gst_no' => $addedInvoice->branch->gst_no,
                'gst' => $addedInvoice->gst,
                'invoiceNumber' => $addedInvoice->invoice_id,
                'invoiceDate' => $addedInvoice->invoice_date ? \Carbon\Carbon::parse($addedInvoice->invoice_date)->format('d-m-Y') : '',
                'dueDate' => $addedInvoice->due_date ? \Carbon\Carbon::parse($addedInvoice->due_date)->format('d-m-Y') : '',
                'discount' => $discount,
                'products' => $products,
                'type' => 'I',
                'payment_terms' => $addedInvoice->payments->isNotEmpty() && $addedInvoice->payments->first()->mode
                    ? $addedInvoice->payments->first()->mode->name
                    : 'N/A',
                'totalQty' => $sumQty,
                'terms' => $addedInvoice->branch->declaration ?? '',
                'redeemed_points' => $redeemed_points,
                'redeemed_value' => $redeemed_value,
                'settings' => $addedInvoice->branch,
            ];

            $html = view('pdf.others', $data)->render();

            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);
            $pdfContent = $mpdf->Output('', 'S');

            // Save to temp file
            $fileName = $addedInvoice->invoice_id . '.pdf';
            $tempPath = storage_path('app/temp/' . $fileName);
            file_put_contents($tempPath, $pdfContent);

            // Create UploadedFile object
            $tempFile = new UploadedFile(
                $tempPath,
                $fileName,
                'application/pdf',
                null,
                true
            );

            // Upload to S3 (Contabo) using your helper
            $filePath = storeFile($tempFile, 'invoices');
            $pdfUrl = getFileUrl($filePath);

            // Clean up local file
            @unlink($tempPath);

            return response()->json([
                'success' => true,
                'pdf_url' => $pdfUrl,
            ]);
        } catch (\Exception $e) {
            Log::error('Error generating invoice: ' . $e->getMessage());
            return response()->json(['error' => 'Something went wrong. Please try again later.'], 500);
        }
    }

    public function destroy($id, Request $request)
    {
        DB::beginTransaction();

        try {
            $invoice = InvoiceModel::where('branch_id', $request->branch_id)
                ->where('id', $id)
                ->firstOrFail();

            $invoice->invoiceItems->each(function ($item) {
                $product = ProductModel::findOrFail($item->product_id);
                $product->qty += $item->quantity;
                $product->save();

                $item->invoiceItemTaxes()->delete();
            });

            $invoice->payments()->delete();
            $invoice->invoiceItems()->delete();
            $invoice->delete();

            DB::commit();

            return response()->json(['message' => 'Invoice deleted successfully!'], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Invoice Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while deleting the invoice.'], 500);
        }
    }


}
