@extends('layouts.layoutapp')

@section('content')

<div class="main-content">
    <div class="row">
        <div class="col-12">
            <div class="card mb-4 mx-4">
                <div class="card-header pb-0">
                    <div class="d-flex flex-row justify-content-between">
                        <div>
                            <h5 class="mb-0">Edit Invoice</h5>
                        </div>
                    </div>
                </div>

                <div class="card-body">
                    <form action="{{ route('invoice-update') }}" method="POST" id="invoiceForm">
                        @csrf
                       <input type="hidden" name="id" value="{{$invoice->id}}">
                        <div class="row">
                            <div class="col-lg-3 mb-3">
                                <label for="client_name" class="form-label">Client Name:</label>
                                <input type="text" name="client_name" id="client_name" class="form-control" 
                                       value="{{ old('client_name', $invoice->client->user->name) }}" readonly>
                            </div>
                            <div class="col-lg-3 mb-3">
                                <label for="client_num" class="form-label">Client Number:</label>
                                <input type="text" name="client_num" id="client_num" class="form-control" 
                                       value="{{ old('client_num', $invoice->client->user->phone) }}" readonly>
                            </div>
                            <div class="col-lg-3 mb-3">
                                <label for="invoice_id" class="form-label">Invoice number:</label>
                                <input type="text" id="invoice_id" class="form-control" value="{{ $invoice->invoice_id }}" readonly>
                            </div>
                            <div class="col-lg-3 mb-3">
                                <label for="invoice_date" class="form-label">Invoice Date:</label>
                                <input type="date" id="invoice_date" class="form-control" name="invoice_date" 
                                       value="{{ old('invoice_date', $invoice->invoice_date) }}" required>
                                @error('invoice_date')
                                    <span class="text-danger">{{ $message }}</span>
                                @enderror
                            </div>
                            <div class="col-lg-3 mb-3">
                                <label for="due_date" class="form-label">Due Date:</label>
                                <input type="date" id="due_date" class="form-control" name="due_date" 
                                       value="{{ old('due_date', $invoice->due_date) }}" required>
                                @error('due_date')
                                    <span class="text-danger">{{ $message }}</span>
                                @enderror
                            </div>
                            <div class="col-lg-3 mb-3">
                                <label for="gst" class="form-label">GST/Non-GST:</label>
                                <select name="gst" id="gst" class="form-select" onchange="toggleTaxes()">
                                    <option value="">Select</option>
                                    <option value="Y" {{ old('gst', $invoice->gst) == 'Y' ? 'selected' : '' }}>GST</option>
                                    <option value="N" {{ old('gst', $invoice->gst) == 'N' ? 'selected' : '' }}>Non-GST</option>
                                </select>
                                @error('gst')
                                    <span class="text-danger">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>
                        <hr>
                        <div>
                            <h5>Product Details:</h5>
                            <div class="row align-items-center">
                                <div class="col-lg-4 d-flex align-items-center mb-3">
                                    <i class="fa-solid fa-qrcode" style="width: 30px; height: 30px; margin-right: 10px;"></i>
                                    <input class="form-control" type="text" id="product-code-input" onkeydown="preventFormSubmit(event)">
                                </div>
                                <div class="col-lg-4">
                                    <button type="button" onclick="addRow(event)" class="btn btn-primary">
                                        <i class="fas fa-plus"></i> Add Row
                                    </button>
                                </div>
                            </div>

                            <div class="table-responsive">
                                <table class="table table-bordered table-striped" id="productsTable">
                                    <thead class="thead-dark">
                                        <tr>
                                            <th>#</th>
                                            <th>Product</th>
                                            <th>Avl. Qty</th>
                                            <th>Qty</th>
                                            <th>Unit Price</th>
                                            <th>Tax</th>
                                            <th>Amount</th>
                                            <th>Action</th>
                                        </tr>
                                    </thead>
                                    <tbody id="productRows">
                                        @foreach($invoice->invoiceItems as $index => $item)
                                            @php
                                                $product = $products->firstWhere('id', $item->product_id);
                                                $currentQty = $product ? $product->qty  : 0;
                                                $taxIds = $item->invoiceItemTaxes->pluck('tax_id')->toArray();
                                               
                                            @endphp
                                            <tr id="row-{{ $index }}">
                                                <td>{{ $index + 1 }}</td>
                                                <td>
                                                    <select name="dynamicFields[{{ $index }}][product]" class="form-select product-select" 
                                                            onchange="updateProductDetails({{ $index }})">
                                                        <option value="">Select Product</option>
                                                        @foreach($products as $product)
                                                            <option value="{{ $product->id }}" 
                                                                    {{ $item->product_id == $product->id ? 'selected' : '' }}>
                                                                {{ $product->code }}-{{ $product->name }}
                                                            </option>
                                                        @endforeach
                                                    </select>
                                                </td>
                                                <td style="width:120px;">
                                                    <input type="text" name="dynamicFields[{{ $index }}][avl_qty]" 
                                                           class="form-control avl-qty" value="{{ $currentQty }}" readonly />
                                                </td>
                                                <td style="width:120px;">
                                                    <input type="text" name="dynamicFields[{{ $index }}][qty]" 
                                                           class="form-control qty" value="{{ $item->quantity }}" 
                                                           onchange="calculateRowAmount({{ $index }})" />
                                                           <div class="invalid-feedback quantity-error" style="display: none;">
                                                            Quantity exceeds available stock!
                                                           </div>
                                                </td>
                                                <td style="width:120px;">
                                                    <input type="text" name="dynamicFields[{{ $index }}][unit_price]" 
                                                           class="form-control unit-price" value="{{ $item->price }}" 
                                                           onchange="calculateRowAmount({{ $index }})" />
                                                </td>
                                                <td style="min-width: 200px;">
                                                    <select name="dynamicFields[{{ $index }}][tax][]" class="form-select tax-select" multiple
                                                            style="height: auto; min-height: 38px;"
                                                            onchange="calculateRowAmount({{ $index }})">
                                                        @foreach($all_taxes as $tax)
                                                            <option value="{{ $tax->id }}" 
                                                                    {{ in_array($tax->id, $taxIds) ? 'selected' : '' }}
                                                                    data-rate="{{ $tax->value }}">
                                                                {{ $tax->name }} - {{ $tax->value }}%
                                                            </option>
                                                        @endforeach
                                                    </select>
                                                </td>
                                                <td style="width:150px;">
                                                    <input type="text" name="dynamicFields[{{ $index }}][amount]" 
                                                           class="form-control amount" value="{{ $item->total }}" readonly />
                                                </td>
                                                <td>
                                                    <button type="button" onclick="removeRow({{ $index }})" class="btn btn-danger btn-sm">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </td>
                                            </tr>
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        <div class="row mt-5">
                            <div class="col-lg-6 mb-3 d-flex">
                                <div class="col-lg-6">
                                    <label for="discount_type" class="form-label">Discount type:</label>
                                    <select name="discount_type" id="discount_type" class="form-select" onchange="calculateTotal()">
                                        <option value="">Select discount type</option>
                                        @foreach($discount_types as $type)
                                            <option value="{{ $type->id }}" 
                                                    {{ old('discount_type', $invoice->discount_type) == $type->id ? 'selected' : '' }}>
                                                {{ $type->type }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('discount_type')
                                        <span class="text-danger">{{ $message }}</span>
                                    @enderror
                                </div>

                                <div class="col-lg-6">
                                    <label for="discount" class="form-label">Discount:</label>
                                    <input type="number" id="discount" name="discount" class="form-control" 
                                           value="{{ old('discount', $invoice->discount) }}" onchange="calculateTotal()" 
                                           {{ empty(old('discount_type', $invoice->discount_type)) ? 'readonly' : '' }}>
                                    @error('discount')
                                        <span class="text-danger">{{ $message }}</span>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-lg-5 mb-3">
                                <div class="card shadow-lg">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between pl-5">
                                            <span>Sub Total:</span>
                                            <span id="subTotalDisplay">{{ number_format($invoice->amount, 2) }} ₹</span>
                                        </div>
                                        <div class="d-flex justify-content-between pl-5">
                                            <span>Discount:</span>
                                            <span id="discountDisplay">{{ number_format($invoice->discount, 2) }} ₹</span>
                                        </div>
                                        <div class="d-flex justify-content-between pl-5">
                                            <span>Tax:</span>
                                            <span id="taxDisplay">{{ number_format($invoice->final_amount - $invoice->amount + $invoice->discount, 2) }} ₹</span>
                                        </div>
                                        <div class="d-flex justify-content-between pl-5">
                                            <span>Total:</span>
                                            <span id="totalDisplay">{{ number_format($invoice->final_amount, 2) }} ₹</span>
                                        </div>
                                        <input type="hidden" id="sub_total" name="sub_total" value="{{ $invoice->amount }}">
                                        <input type="hidden" id="discount_amt" name="discount_amt" value="{{ $invoice->discount }}">
                                        <input type="hidden" id="tax" name="tax" 
                                               value="{{ $invoice->final_amount - $invoice->amount + $invoice->discount }}">
                                        <input type="hidden" id="total_amt" name="total_amt" value="{{ $invoice->final_amount }}">
                                    </div>
                                </div>
                            </div>
                        </div>

                        <hr>
                        <div>
                            <button type="button" onclick="toggleNotesAndTerms()" class="btn btn-primary" id="notesTermsBtn">
                                @if($invoice->note || $invoice->term)
                                    <i class="fas fa-minus"></i> remove Note & Term
                                @else
                                    <i class="fas fa-plus"></i> Add Note & Term
                                @endif
                            </button>
                            <div class="row" id="notesTermsSection" 
                                 style="display: {{ $invoice->note || $invoice->term ? 'flex' : 'none' }};">
                                <div class="col-lg-6">
                                    <label for="note" class="form-label">Note:</label>
                                    <textarea class="form-control" name="note" id="note">{{ old('note', $invoice->note) }}</textarea>
                                </div>
                                <div class="col-lg-6">
                                    <label for="term" class="form-label">Term:</label>
                                    <textarea class="form-control" name="term" id="term">{{ old('term', $invoice->term) }}</textarea>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="submit" class="btn btn-primary">Update</button>
                            <a href="{{ route('invoice-index') }}" class="btn btn-secondary">Cancel</a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    // Product data from backend
    const products = @json($products);
    const taxes = @json($all_taxes);
    const defaultTaxIds = @json($default_taxes);        

    let rowCount = {{ count($invoice->invoiceItems) }};
    let hasQuantityError = false;
    
    function preventFormSubmit(event) {
        if (event.key === 'Enter') {
            event.preventDefault();
            addProductRow();
            setTimeout(() => {
                document.getElementById('product-code-input').focus();
            }, 1000);
        }
    }
    
    function addRow(event) {
        event.preventDefault();
        addProductRow();
        setTimeout(() => {
            document.getElementById('product-code-input').focus();
        }, 1000);
    }
    
    function addProductRow() {
        const productCode = document.getElementById('product-code-input').value;
        if (!productCode) {
            addEmptyRow();
            return;
        }
        
        const product = products.find(p => p.code === productCode);
        if (product) {
            const existingRow = findProductRow(product.id);
            
            if (existingRow) {
                const qtyInput = existingRow.querySelector('.qty');
                const currentQty = parseFloat(qtyInput.value) || 0;
                qtyInput.value = currentQty + 1;
                calculateRowAmount(existingRow.id.split('-')[1]); 
                checkQuantity(existingRow.id.split('-')[1]);
            } else {
                addProductToTable(product);
            }
            
            document.getElementById('product-code-input').value = '';
        } else {
            Swal.fire({
                icon: 'error',
                title: 'Product Not Found',
                text: 'The product code you entered was not found in the system',
                confirmButtonText: 'OK'
            });
        }
    }

    function findProductRow(productId) {
        const rows = document.querySelectorAll('#productRows tr');
        for (const row of rows) {
            const select = row.querySelector('.product-select');
            if (select && select.value == productId) {
                return row;
            }
        }
        return null;
    }
    
    function addEmptyRow() {
        rowCount++;
        const row = `
            <tr id="row-${rowCount}">
                <td>${rowCount}</td>
                <td>
                    <select name="dynamicFields[${rowCount}][product]" class="form-select product-select" 
                            onchange="updateProductDetails(${rowCount})">
                        <option value="">Select Product</option>
                        ${products.map(p => `<option value="${p.id}">${p.code}-${p.name}</option>`).join('')}
                    </select>
                </td>
                <td style="width:120px;">
                    <input type="text" name="dynamicFields[${rowCount}][avl_qty]" class="form-control avl-qty" readonly />
                </td>
                <td style="width:120px;">
                    <input type="text" name="dynamicFields[${rowCount}][qty]" class="form-control qty" 
                           onchange="calculateRowAmount(${rowCount})" 
                           oninput="checkQuantity(${rowCount})" />
                    <div class="invalid-feedback quantity-error" style="display: none;">
                        Quantity exceeds available stock!
                    </div>
                </td>
                <td style="width:120px;">
                    <input type="text" name="dynamicFields[${rowCount}][unit_price]" class="form-control unit-price"
                           onchange="calculateRowAmount(${rowCount})" />
                </td>
              <td style="min-width: 200px;">
                <select name="dynamicFields[${rowCount}][tax][]" class="form-select tax-select" multiple
                        style="height: auto; min-height: 38px;"
                        onchange="calculateRowAmount(${rowCount})">
                    ${taxes.map(t => {
                        return `<option value="${t.id}">${t.name} - ${t.value}%</option>`;
                    }).join('')}
                </select>
            </td>
                <td style="width:150px;">
                    <input type="text" name="dynamicFields[${rowCount}][amount]" class="form-control amount" readonly />
                </td>
                <td>
                    <button type="button" onclick="removeRow(${rowCount})" class="btn btn-danger btn-sm">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `;
        document.getElementById('productRows').insertAdjacentHTML('beforeend', row);
        $(`#row-${rowCount} .product-select`).select2({
        placeholder: "Select Product",
        width: '100%'
    });
        // If GST is selected, enable taxes, otherwise disable
        const gstValue = document.getElementById('gst').value;
        if (gstValue === 'N') {
            const taxSelect = document.querySelector(`#row-${rowCount} .tax-select`);
            taxSelect.innerHTML = '<option value="no-tax" selected>No Tax</option>';
            taxSelect.disabled = true;
        }
    }

    function addProductToTable(product) {
    rowCount++;
    const row = `
        <tr id="row-${rowCount}">
            <td>${rowCount}</td>
            <td>
                <select name="dynamicFields[${rowCount}][product]" class="form-select product-select" 
                        onchange="updateProductDetails(${rowCount})">
                    <option value="">Select Product</option>
                    ${products.map(p => `<option value="${p.id}" ${p.id == product.id ? 'selected' : ''}>${p.code}-${p.name}</option>`).join('')}
                </select>
            </td>
            <td style="width:120px;">
                <input type="text" name="dynamicFields[${rowCount}][avl_qty]" class="form-control avl-qty" 
                       value="${product.qty}" readonly />
            </td>
            <td style="width:120px;">
                <input type="text" name="dynamicFields[${rowCount}][qty]" class="form-control qty" 
                       value="1" onchange="calculateRowAmount(${rowCount})" 
                       oninput="checkQuantity(${rowCount})" />
                <div class="invalid-feedback quantity-error" style="display: none;">
                    Quantity exceeds available stock!
                </div>
            </td>
            <td style="width:120px;">
                <input type="text" name="dynamicFields[${rowCount}][unit_price]" class="form-control unit-price" 
                       value="${product.unit_price}" onchange="calculateRowAmount(${rowCount})" />
            </td>
            <td style="min-width: 200px;">
                <select name="dynamicFields[${rowCount}][tax][]" class="form-select tax-select" multiple
                        style="height: auto; min-height: 38px;"
                        onchange="calculateRowAmount(${rowCount})">
                    ${taxes.map(t => {
                        // Only select if it's in the product's taxes or default taxes
                        const isProductTax = product.taxes ? product.taxes.includes(t.id) : false;
                        return `<option value="${t.id}" ${isProductTax ? 'selected' : ''}>${t.name} - ${t.value}%</option>`;
                    }).join('')}
                </select>
            </td>
            <td style="width:150px;">
                <input type="text" name="dynamicFields[${rowCount}][amount]" class="form-control amount" 
                       value="${product.unit_price}" readonly />
            </td>
            <td>
                <button type="button" onclick="removeRow(${rowCount})" class="btn btn-danger btn-sm">
                    <i class="fas fa-trash"></i>
                </button>
            </td>
        </tr>
    `;
    document.getElementById('productRows').insertAdjacentHTML('beforeend', row);
    $(`#row-${rowCount} .product-select`).select2({
        placeholder: "Select Product",
        width: '100%'
    });
    calculateRowAmount(rowCount);
    checkQuantity(rowCount);
    
    // If GST is selected, enable taxes, otherwise disable
    const gstValue = document.getElementById('gst').value;
    if (gstValue === 'N') {
        const taxSelect = document.querySelector(`#row-${rowCount} .tax-select`);
        taxSelect.innerHTML = '<option value="no-tax" selected>No Tax</option>';
        taxSelect.disabled = true;
    }
}
    
    function updateProductDetails(rowId) {
        const row = document.getElementById(`row-${rowId}`);
        const productId = row.querySelector('.product-select').value;
        const product = products.find(p => p.id == productId);
        
        if (product) {
            row.querySelector('.avl-qty').value = product.qty;
            row.querySelector('.qty').value = 1;
            row.querySelector('.unit-price').value = product.unit_price;
            calculateRowAmount(rowId);
            checkQuantity(rowId);
        } else {
            row.querySelector('.avl-qty').value = '';
            row.querySelector('.qty').value = '';
            row.querySelector('.unit-price').value = '';
            calculateRowAmount(rowId);
        }
    }
    
    function checkQuantity(rowId) {
        const row = document.getElementById(`row-${rowId}`);
        if (!row) return true; // Row might have been removed
        
        const qtyInput = row.querySelector('.qty');
        const avlQty = parseFloat(row.querySelector('.avl-qty').value) || 0;
        const qty = parseFloat(qtyInput.value) || 0;
        const errorElement = row.querySelector('.quantity-error');
        
        if (qty > avlQty) {
            qtyInput.classList.add('is-invalid');
            errorElement.style.display = 'block';
            hasQuantityError = true;
            return false;
        } else {
            qtyInput.classList.remove('is-invalid');
            errorElement.style.display = 'none';
            
            // Check all rows for errors
            hasQuantityError = false;
            document.querySelectorAll('#productRows tr').forEach(r => {
                const rQtyInput = r.querySelector('.qty');
                const rAvlQty = parseFloat(r.querySelector('.avl-qty').value) || 0;
                const rQty = parseFloat(rQtyInput?.value) || 0;
                
                if (rQty > rAvlQty) {
                    hasQuantityError = true;
                }
            });
            
            return true;
        }
    }
    
    function calculateRowAmount(rowId) {
        const row = document.getElementById(`row-${rowId}`);
        if (!row) return; // Row might have been removed
        
        const qty = parseFloat(row.querySelector('.qty').value) || 0;
        const unitPrice = parseFloat(row.querySelector('.unit-price').value) || 0;
        const amount = qty * unitPrice;
        
        // Calculate tax
        const taxSelect = row.querySelector('.tax-select');
        const selectedTaxes = Array.from(taxSelect.selectedOptions).map(opt => parseInt(opt.value));
        let taxRate = 0;
        
        selectedTaxes.forEach(taxId => {
            if (taxId && taxId !== 'no-tax') {
                const tax = taxes.find(t => t.id == taxId);
                if (tax) taxRate += tax.value;
            }
        });
        
        const taxAmount = amount * (taxRate / 100);
        const totalWithTax = amount + taxAmount;
        
        row.querySelector('.amount').value = amount.toFixed(2);
        calculateTotal();
    }
    
    function calculateTotal() {
        let subTotal = 0;
        let totalTax = 0;
        
        document.querySelectorAll('#productRows tr').forEach(row => {
            const amount = parseFloat(row.querySelector('.amount').value) || 0;
            const qty = parseFloat(row.querySelector('.qty').value) || 0;
            const unitPrice = parseFloat(row.querySelector('.unit-price').value) || 0;
            const rowAmount = qty * unitPrice;
            
            subTotal += rowAmount;
            
            // Calculate tax for this row
            const taxSelect = row.querySelector('.tax-select');
            const selectedTaxes = Array.from(taxSelect.selectedOptions).map(opt => parseInt(opt.value));
            let taxRate = 0;
            
            selectedTaxes.forEach(taxId => {
                if (taxId && taxId !== 'no-tax') {
                    const tax = taxes.find(t => t.id == taxId);
                    if (tax) taxRate += tax.value;
                }
            });
            
            totalTax += rowAmount * (taxRate / 100);
        });
        
        // Calculate discount
        const discountType = document.getElementById('discount_type').value;
        const discountValue = parseFloat(document.getElementById('discount').value) || 0;
        let discountAmount = 0;

        var subtotalWithTax = subTotal + totalTax;
        
        if (discountType == 1) {
            discountAmount = discountValue;
        } else if (discountType == 2) {
            discountAmount = subtotalWithTax * (discountValue / 100);
        }
        
        const total = subtotalWithTax - discountAmount;
        
        // Update display
        document.getElementById('subTotalDisplay').textContent = subTotal.toFixed(2) + ' ₹';
        document.getElementById('discountDisplay').textContent = discountAmount.toFixed(2) + ' ₹';
        document.getElementById('taxDisplay').textContent = totalTax.toFixed(2) + ' ₹';
        document.getElementById('totalDisplay').textContent = Math.floor(total).toFixed(2) + ' ₹';
        
        // Update hidden fields
        document.getElementById('sub_total').value = subTotal.toFixed(2);
        document.getElementById('discount_amt').value = discountAmount.toFixed(2);
        document.getElementById('tax').value = totalTax.toFixed(2);
        document.getElementById('total_amt').value = Math.floor(total).toFixed(2);
    }
    
    function toggleTaxes() {
        const gstValue = document.getElementById('gst').value;
        const taxSelects = document.querySelectorAll('.tax-select');

        taxSelects.forEach(select => {
            if (gstValue === 'N') {
                select.innerHTML = '<option value="no-tax" selected>No Tax</option>';
                select.disabled = true;
            } else {
                select.disabled = false;
                select.innerHTML = taxes.map(t => {
                    const isDefault = defaultTaxIds.includes(t.id);
                    return `<option value="${t.id}" ${isDefault ? 'selected' : ''}>${t.name} - ${t.value}%</option>`;
                }).join('');
            }
        });

        calculateTotal();
    }
    
    function toggleNotesAndTerms() {
        const section = document.getElementById('notesTermsSection');
        const btn = document.getElementById('notesTermsBtn');
        
        if (section.style.display === 'none') {
            section.style.display = 'flex';
            btn.innerHTML = '<i class="fas fa-minus"></i> remove Note & Term';
        } else {
            section.style.display = 'none';
            btn.innerHTML = '<i class="fas fa-plus"></i> Add Note & Term';
        }
    }
    
    function removeRow(rowId) {
        const row = document.getElementById(`row-${rowId}`);
        if (row) {
            row.remove();
            
            // Renumber remaining rows
            let index = 1;
            document.querySelectorAll('#productRows tr').forEach(row => {
                row.cells[0].textContent = index;
                index++;
            });
            
            calculateTotal();
            
            // Check for quantity errors after removal
            hasQuantityError = false;
            document.querySelectorAll('#productRows tr').forEach(r => {
                const rQtyInput = r.querySelector('.qty');
                const rAvlQty = parseFloat(r.querySelector('.avl-qty').value) || 0;
                const rQty = parseFloat(rQtyInput?.value) || 0;
                
                if (rQty > rAvlQty) {
                    hasQuantityError = true;
                }
            });
        }
    }
    
    function validateForm(event) {
        // First check all quantities
        hasQuantityError = false;
        document.querySelectorAll('#productRows tr').forEach(row => {
            const rowId = row.id.split('-')[1];
            checkQuantity(rowId);
        });
        
        if (hasQuantityError) {
            event.preventDefault();
            Swal.fire({
                icon: 'error',
                title: 'Quantity Error',
                text: 'Please fix quantity errors before submitting. Some items have quantity greater than available stock.',
                confirmButtonText: 'OK',
            });
            
            // Scroll to first error
            const firstError = document.querySelector('.is-invalid');
            if (firstError) {
                firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
            
            return false;
        }
        
        return true;
    }
    
    $(document).ready(function() {

        $('.product-select').select2({
        placeholder: "Select Product",
        width: '100%'
    });
        if (!document.getElementById('invoice_date').value) {
            document.getElementById('invoice_date').valueAsDate = new Date();
        }
        if (!document.getElementById('due_date').value) {
            document.getElementById('due_date').valueAsDate = new Date();
        }
        
        document.getElementById('discount_type').addEventListener('change', function() {
            document.getElementById('discount').readOnly = !this.value;
            calculateTotal();
        });

        document.querySelector('form').addEventListener('submit', validateForm);
        
        toggleTaxes();
        
        calculateTotal();
        
        document.querySelectorAll('#productRows tr').forEach(row => {
            const rowId = row.id.split('-')[1];
            checkQuantity(rowId);
        });
    });
    
    $('#gst').on('change', function() {
        toggleTaxes();
    });
</script>
@endpush