<?php

namespace App\Imports;

use App\Models\ClientModel;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Maatwebsite\Excel\Concerns\RemembersRowNumber;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Events\AfterImport;
use Illuminate\Support\Facades\Log;

class ClientImport implements ToModel, WithHeadingRow, WithEvents
{
    use RemembersRowNumber;

    protected $row;
    protected $hasErrors = false;

    public function registerEvents(): array
    {
        return [
            BeforeImport::class => function (BeforeImport $event) {
                DB::beginTransaction();
            },
            AfterImport::class => function (AfterImport $event) {
                $errors = session()->get('client_err', []);
                if (empty($errors)) {
                    DB::commit();
                } else {
                    DB::rollBack();
                }
            }
        ];
    }

    public function model(array $row)
    {
        $this->row = $this->getRowNumber();
        $this->hasErrors = false;

        // Validate the row data first
        $this->validateRequiredFields($row);
        $this->checkForDuplicates($row);
        $this->validatePasswords($row);

        // If there are errors, don't proceed with creation
        if ($this->hasErrors) {
            return null;
        }

        try {
            // Create user
            $user = User::create([
                'name' => $row['first_name'] ?? '',
                'last_name' => $row['last_name'] ?? null,
                'email' => $row['email'] ?? '',
                'phone' => $row['contact_number'] ?? '',
                'password' => Hash::make($row['password'] ?? ''),
                'user_flg' => 'C',
                'active_status' => $row['active_status'] ?? 'A',
                'branch_id' => session('branch_id') ?? null,
            ]);

            // Create client
            return new ClientModel([
                'user_id' => $user->id,
                'website' => $row['website'] ?? null,
                'postal_code' => $row['postal_code'] ?? null,
                'country_id' => $this->getCountry($row['country'] ?? null),
                'state_id' => $this->getState($row['state'] ?? null),
                'city_id' => $this->getCity($row['city'] ?? null),
                'company_name' => $row['company_name'] ?? null,
                'address' => $row['address'] ?? null,
                'note' => $row['notes'] ?? null,
                'gstin' => $row['gst_in'] ?? '',
                'branch_id' => session('branch_id') ?? null,
                'active_status' => $row['active_status'] ?? 'A',
            ]);

        } catch (\Exception $e) {
            $this->storeErrors('System Error', '', 'Failed to create client: ' . $e->getMessage());
            Log::error('While importing the Client at row ' . $this->row . ': ' . $e->getMessage());
            return null;
        }
    }

    private function validateRequiredFields(array $row)
    {
        $requiredFields = [
            'contact_number' => 'Contact Number',
            'gst_in' => 'GST IN',
            'first_name' => 'First Name',
            'email' => 'Email',
            'password' => 'Password',
            'confirm_password' => 'Confirm Password',
            // 'city' => 'City'
        ];

        foreach ($requiredFields as $field => $name) {
            if (empty($row[$field])) {
                $this->storeErrors($name, $row[$field] ?? '', 'This field is required');
                $this->hasErrors = true;
                return null;
            }
        }
    }

    private function checkForDuplicates(array $row)
    {
        if (!empty($row['contact_number'])) {
            if (!is_numeric($row['contact_number'])) {
                $this->storeErrors('Contact Number', $row['contact_number'], 'Must be a numeric value');
                $this->hasErrors = true;
            } elseif (strlen($row['contact_number']) != 10) {
                $this->storeErrors('Contact Number', $row['contact_number'], 'Must be exactly 10 digits');
                $this->hasErrors = true;
            } elseif (User::where('phone', $row['contact_number'])->exists()) {
                $this->storeErrors('Contact Number', $row['contact_number'], 'Already Exists');
                $this->hasErrors = true;
            }
        }
        if (!empty($row['gst_in']) && ClientModel::where('gstin', $row['gst_in'])->exists()) {
            $this->storeErrors('GST IN', $row['gst_in'], 'Already Exists');
            $this->hasErrors = true;
        }

        if (!empty($row['email']) && User::where('email', $row['email'])->exists()) {
            $this->storeErrors('Email', $row['email'], 'Already Exists');
            $this->hasErrors = true;
        }

    }

    private function validatePasswords(array $row)
    {
        if (empty($row['password'])) {
            $this->storeErrors('Password', '', 'Password is required');
            $this->hasErrors = true;
        }

        if (empty($row['confirm_password'])) {
            $this->storeErrors('Confirm Password', '', 'Confirm Password is required');
            $this->hasErrors = true;
        }

        if (
            !empty($row['password']) && !empty($row['confirm_password']) &&
            $row['password'] !== $row['confirm_password']
        ) {
            $this->storeErrors('Password', '', 'Password and Confirm Password must match');
            $this->hasErrors = true;
        }
    }

    private function getCountry($data)
    {

        $country = DB::table('countries')->where('name', $data)->value('id');
        if (empty($country)) {
            $this->storeErrors('Country', $data, 'Unable to find correct the spelling.');
            $this->hasErrors = true;
            return null;
        }
        dump($country);
        return $country;
    }

    private function getState($data)
    {
        $state = DB::table('states')->where('name', $data)->value('id');
        if (empty($state)) {
            $this->storeErrors('State', $data, 'Unable to find correct the spelling.');
            $this->hasErrors = true;
            return null;
        }
        dump($state);

        return $state;
    }

    private function getCity($data)
    {
        $city = DB::table('cities')->where('name', $data)->value('id');
        if (empty($city)) {
            $this->storeErrors('City', $data, 'Unable to find correct the spelling.');
            $this->hasErrors = true;
            return null;
        }

        return $city;
    }

    private function storeErrors($key, $value = '', $error = 'Invalid value')
    {
        $errorMessage = htmlspecialchars($key) . ': ' .
            htmlspecialchars($value) . ' - ' .
            htmlspecialchars($error) . ' at row ' . $this->row;

        $errors = session()->get('client_err', []);
        $errors[] = $errorMessage;
        session()->put('client_err', $errors);
    }
}